/* prefs.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported ExtensionTemperaturePreferences */

import Adw from 'gi://Adw';
import Gio from 'gi://Gio';
import GObject from 'gi://GObject';
import Gtk from 'gi://Gtk';

import { ExtensionPreferences, gettext as _ } from 'resource:///org/gnome/Shell/Extensions/js/extensions/prefs.js';

function getPreferences(group) {
    let prefGroup = null;
    switch (group) {
    case 'fahrenheit':
        prefGroup = {
            prefs: [
                {
                    type: 'switcher',
                    key: 'show-fahrenheit',
                    title: _('Show temperature in Fahrenheit'),
                    subtitle: _('When this option is activated, Fahrenheit is used as the temperature unit for the display instead of Celsius.'),
                },
            ],
        };
        break;
    case 'refresh':
        prefGroup = {
            prefs: [
                {
                    type: 'combo',
                    key: 'refresh-interval',
                    title: _('Refresh interval in seconds'),
                    subtitle: _('Sets the refresh interval for the temperature query of the sensors in seconds.'),
                    values: {
                        '10': '10',
                        '15': '15',
                        '30': '30',
                        '60': '60',
                        '120': '120',
                        '300': '300',
                        '600': '600',
                    },
                },
            ],
        };
        break;
    default: // Nothing to do
    }
    return prefGroup;
}

const TemperaturePreferences = GObject.registerClass(
class TemperaturePreferences extends Adw.PreferencesPage {
    _init(settings) {
        super._init();
        this._settings = settings;
        this._createPrefGroup(getPreferences('fahrenheit'));
        this._createPrefGroup(getPreferences('refresh'));
    }

    _createPrefGroup(group) {
        if (group === null) return;
        let prefGroup = new Adw.PreferencesGroup({ title: group?.title ?? null });
        group.prefs.forEach(pref => {
            let prefWidget = this._createPrefWidget(pref);
            if (prefWidget !== null) prefGroup.add(prefWidget);
        });
        this.add(prefGroup);
    }

    _createPrefWidget(pref) {
        let prefAction = null;
        let prefWidget = null;
        let settings = pref.schema ? new Gio.Settings({ schema_id: pref.schema }) : this._settings;
        switch (pref.type) {
        case 'combo':
            let model = new Gtk.StringList();
            for (let value in pref.values) {
                model.append(pref.values[value]);
            }
            prefAction = new Gtk.DropDown({
                model: model,
                valign: Gtk.Align.CENTER,
                width_request: 100,
            });
            prefAction.connect('notify::selected-item', () => {
                settings.set_enum(pref.key, prefAction.get_selected());
            });
            prefAction.set_selected(settings.get_enum(pref.key));
            settings.connect('changed::%s'.format(pref.key), () => {
                prefAction.set_selected(settings.get_enum(pref.key));
            });
            break;
        case 'switcher':
            prefAction = new Gtk.Switch({
                active: settings.get_boolean(pref.key),
                valign: Gtk.Align.CENTER,
            });
            settings.bind(pref.key, prefAction, 'active', Gio.SettingsBindFlags.DEFAULT);
            break;
        default: // Nothing to do
        }
        if (prefAction !== null) {
            prefWidget = new Adw.ActionRow({
                title: pref.title,
                subtitle: pref?.subtitle ?? null,
            });
            prefWidget.add_suffix(prefAction);
            prefWidget.set_activatable_widget(prefAction);
        }
        return prefWidget;
    }
});

export default class ExtensionTemperaturePreferences extends ExtensionPreferences {
    fillPreferencesWindow(window) {
        window.add(new TemperaturePreferences(this.getSettings()));
    }
}
