/* systemMenu.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported SystemMenuButton */

import Gio from 'gi://Gio';
import GLib from 'gi://GLib';
import GObject from 'gi://GObject';
import Shell from 'gi://Shell';
import St from 'gi://St';

import { Extension, gettext as _ } from 'resource:///org/gnome/shell/extensions/extension.js';

import * as LoginManager from 'resource:///org/gnome/shell/misc/loginManager.js';
import * as Util from 'resource:///org/gnome/shell/misc/util.js';

import * as Main from 'resource:///org/gnome/shell/ui/main.js';
import * as PopupMenu from 'resource:///org/gnome/shell/ui/popupMenu.js';
import * as QuickSettings from 'resource:///org/gnome/shell/ui/quickSettings.js';

const BASIC_SETTINGS = [
    'ca.desrt.dconf-editor.desktop',        // dconf-editor
    'org.gnome.Extensions.desktop',         // gnome-shell-extension-prefs
    'org.gnome.Software.desktop',           // gnome-software
    'org.gnome.tweaks.desktop',             // gnome-tweaks
    'system-config-printer.desktop',        // system-config-printer
    'synaptic.desktop',                     // synaptic
    'users.desktop'                         // gnome-system-tools
]

const GNOME_TERMINAL = 'org.gnome.Terminal.desktop';

const KEY_DISABLE_RESTART_BUTTONS = 'disable-restart-buttons';
const KEY_SHOW_BASIC_SETTINGS = 'show-basic-settings';
const KEY_SHOW_GNOME_TERMINAL = 'show-gnome-terminal';
const KEY_SHOW_HIBERNATE = 'show-hibernate';
const KEY_SHOW_HYBRID_SLEEP = 'show-hybrid-sleep';
const KEY_SHOW_SYSTEMMENU = 'show-systemmenu';

const LOGIN_SCREEN_SCHEMA = 'org.gnome.login-screen';

const N_QUICK_SETTINGS_COLUMNS = 2;

const SYSTEMMENU_ICON = 1;

const QuickPreferencesSettingsItem = GObject.registerClass(
class QuickPreferencesSettingsItem extends QuickSettings.QuickSettingsItem {
    _init(uuid) {
        super._init({
            style_class: 'icon-button',
            can_focus: true,
            icon_name: 'preferences-other-symbolic',
            accessible_name: _('Settings Panel Manager'),
        });
        this.connect('clicked', () => {
            Main.overview.hide();
            Main.panel.closeQuickSettings();
            Util.spawn([ 'gnome-extensions', 'prefs', uuid ]);
        });
    }
});

const QuickTerminalSettingsItem = GObject.registerClass(
class QuickTerminalSettingsItem extends QuickSettings.QuickSettingsItem {
    _init(settings) {
        super._init({
            style_class: 'icon-button',
            can_focus: true,
            child: new St.Icon(),
        });
        this._settings = settings;
        this._settingsChangedId = this._settings.connect('changed::%s'.format(KEY_SHOW_GNOME_TERMINAL), this._onChangedShowTerminal.bind(this));
        this._appSystem = Shell.AppSystem.get_default();
        this._appSystemChangedId = this._appSystem.connect('installed-changed', this._onInstalledChanged.bind(this));
        this._onInstalledChanged();
        this.connect('clicked', () => {
            Main.overview.hide();
            Main.panel.closeQuickSettings();
            this._terminalApp.activate();
        });
        this.connect('destroy', () => {
            this._appSystem.disconnect(this._appSystemChangedId);
        });
    }

    _onChangedShowTerminal() {
        let showTerminal = this._settings.get_boolean(KEY_SHOW_GNOME_TERMINAL);
        this.visible = this._terminalApp !== null && showTerminal;
    }

    _onInstalledChanged() {
        this._terminalApp = this._appSystem.lookup_app(GNOME_TERMINAL);
        this.child.gicon = this._terminalApp?.get_icon() ?? null;
        this.accessible_name = this._terminalApp?.get_name() ?? null;
        this._onChangedShowTerminal();
    }
});

class PopupHibernationMenuSection extends PopupMenu.PopupMenuSection {
    constructor(settings) {
        super();
        this._settings = settings;
        this._settingsChangedId = [
            this._settings.connect('changed::%s'.format(KEY_SHOW_HYBRID_SLEEP), this._updateHibernation.bind(this)),
            this._settings.connect('changed::%s'.format(KEY_SHOW_HIBERNATE), this._updateHibernation.bind(this)),
        ];
        this._loginManager = LoginManager.getLoginManager();
        this._loginScreenSettings = new Gio.Settings({ schema_id: LOGIN_SCREEN_SCHEMA });
        this._itemHybridSleep = new PopupMenu.PopupMenuItem(_('Hybrid Sleep'));
        this._itemHybridSleep.connect('activate', () => {
            this._onHibernation('HybridSleep');
        });
        this.addMenuItem(this._itemHybridSleep);
        this._itemHibernate = new PopupMenu.PopupMenuItem(_('Hibernate'));
        this._itemHibernate.connect('activate', () => {
            this._onHibernation('Hibernate');
        });
        this.addMenuItem(this._itemHibernate);
        this._openStateChangedId = Main.panel.statusArea.quickSettings.menu.connect('open-state-changed', (menu, open) => {
            if (!open) return;
            this._updateHibernation();
        });
    }

    _canHibernation(value, asyncCallback) {
        if (this._loginManager._proxy) {
            this._loginManager._proxy.call(value, null, Gio.DBusCallFlags.NONE, -1, null, (proxy, asyncResult) => {
                try {
                    let result = proxy.call_finish(asyncResult).deep_unpack();
                    asyncCallback(!['no', 'na'].includes(result[0]));
                } catch (e) {
                    asyncCallback(false);
                }
            });
        } else {
            Mainloop.idle_add(() => {
                asyncCallback(false);
                return false;
            });
        }
    }

    _onHibernation(value) {
        if (this._loginManager._proxy) {
            this._loginManager._proxy.call(value, GLib.Variant.new('(b)', [true]), Gio.DBusCallFlags.NONE, -1, null, null);
        } else {
            this._loginManager.emit('prepare-for-sleep', true);
            this._loginManager.emit('prepare-for-sleep', false);
        }
        Main.overview.hide();
        Main.panel.closeQuickSettings();
    }

    _updateHibernation() {
        let disableRestartButtons = this._loginScreenSettings.get_boolean(KEY_DISABLE_RESTART_BUTTONS);
        let disabled = Main.sessionMode.isLocked || (Main.sessionMode.isGreeter && disableRestartButtons);
        this._canHibernation('CanHibernate', (result) => {
            this._itemHibernate.visible = result && !disabled && this._settings.get_boolean(KEY_SHOW_HIBERNATE);
        });
        this._canHibernation('CanHybridSleep', (result) => {
            this._itemHybridSleep.visible = result && !disabled && this._settings.get_boolean(KEY_SHOW_HYBRID_SLEEP);
        });
        this.visible = this._itemHibernate.visible || this._itemHybridSleep.visible;
    }

    destroy() {
        Main.panel.statusArea.quickSettings.menu.disconnect(this._openStateChangedId);
        this._settingsChangedId.forEach(id => {
            this._settings.disconnect(id);
        });
        super.destroy();
    }
}

class SystemIndicator {
    constructor(uuid, settings) {
        this._preferencesItem = new QuickPreferencesSettingsItem(uuid);
        Main.panel.statusArea.quickSettings._system._systemItem.child.insert_child_below(this._preferencesItem, Main.panel.statusArea.quickSettings._system._systemItem._desktopSpacer);
        this._terminalItem = new QuickTerminalSettingsItem(settings);
        Main.panel.statusArea.quickSettings._system._systemItem.child.insert_child_above(this._terminalItem, Main.panel.statusArea.quickSettings._system._systemItem._desktopSpacer);
        this._hibernationSection = new PopupHibernationMenuSection(settings);
        Main.panel.statusArea.quickSettings._system._systemItem.menu.addMenuItem(this._hibernationSection, 1);
    }

    destroy() {
        this._hibernationSection.destroy();
        this._terminalItem.destroy();
        this._preferencesItem.destroy();
    }
}

const SystemBasicSettingsIndicator = GObject.registerClass(
class SystemBasicSettingsIndicator extends QuickSettings.SystemIndicator {
    _init(settings) {
        super._init();
        this._basicSettings = new QuickSettings.QuickMenuToggle({
            title: _('Basic Settings'),
            iconName: 'preferences-system-symbolic',
        });
        this._basicSettings.menu.setHeader('preferences-system-symbolic', _('Basic Settings'));
        this.quickSettingsItems.push(this._basicSettings);
        this._settings = settings;
        this._settingsChangedId = this._settings.connect('changed::%s'.format(KEY_SHOW_BASIC_SETTINGS), this._onChangedShowBasicSettings.bind(this));
        this._appSystem = Shell.AppSystem.get_default();
        this._appSystemChangedId = this._appSystem.connect('installed-changed', this._reloadBasicSettings.bind(this));
        this._reloadBasicSettings();
        this._basicSettings.connect('clicked', () => this._basicSettings.menu.open());
        this.connect('destroy', () => {
            this._appSystem.disconnect(this._appSystemChangedId);
            this._basicSettings.destroy();
        });
    }

    _onChangedShowBasicSettings() {
        let showBasicSettings = this._settings.get_boolean(KEY_SHOW_BASIC_SETTINGS);
        this._basicSettings.visible = this._basicSettings.menu.numMenuItems > 0 && showBasicSettings;
    }

    _reloadBasicSettings() {
        this._basicSettings.menu.removeAll();
        let appList = [];
        BASIC_SETTINGS.forEach(desktopFile => {
            let app = this._appSystem.lookup_app(desktopFile);
            if (app !== null) appList.push(app);
        }, this);
        appList.sort((a, b) => {
            return a.compare_by_name(b);
        });
        appList.forEach(app => {
            let item = new PopupMenu.PopupMenuItem(app.get_name());
            item.connect('activate', () => {
                app.activate();
                Main.overview.hide();
                Main.panel.closeQuickSettings();
            });
            this._basicSettings.menu.addMenuItem(item);
        });
        this._onChangedShowBasicSettings();
    }
});

export class SystemMenuButton {
    constructor(uuid, settings) {
        this._currentChild = Main.panel.statusArea.quickSettings._indicators;
        this._icon = new St.Icon({
            style_class: 'system-status-icon',
            icon_name: 'system-shutdown-symbolic',
        });
        this._settings = settings;
        this._settingsChangedId = this._settings.connect('changed::%s'.format(KEY_SHOW_SYSTEMMENU), this._onChangedShowSystemMenu.bind(this));
        this._onChangedShowSystemMenu();
        this._system = new SystemIndicator(uuid, settings);
        this._basicSettings = new SystemBasicSettingsIndicator(settings);
        Main.panel.statusArea.quickSettings._indicators.add_child(this._basicSettings);
        this._basicSettings.quickSettingsItems.forEach(item => Main.panel.statusArea.quickSettings.menu.addItem(item, N_QUICK_SETTINGS_COLUMNS));
    }

    _onChangedShowSystemMenu() {
        let showSystemMenu = this._settings.get_enum(KEY_SHOW_SYSTEMMENU);
        if (showSystemMenu === SYSTEMMENU_ICON) {
            this._setChild(this._icon);
        } else {
            this._setChild(Main.panel.statusArea.quickSettings._indicators);
        }
    }

    _setChild(child) {
        if (!child || child === this._currentChild) return;
        let current = Main.panel.statusArea.quickSettings.get_children()[0];
        Main.panel.statusArea.quickSettings.remove_child(current);
        Main.panel.statusArea.quickSettings.add_child(child);
        this._currentChild = child;
    }

    destroy() {
        this._basicSettings.destroy();
        this._system.destroy();
        this._settings.disconnect(this._settingsChangedId);
        this._setChild(Main.panel.statusArea.quickSettings._indicators);
        this._icon.destroy();
    }
}
