/* prefs.js
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 */

/* exported init buildPrefsWidget */

const Gettext = imports.gettext;

const { Gio, GObject, Gtk } = imports.gi;

const ExtensionUtils = imports.misc.extensionUtils;

const Me = ExtensionUtils.getCurrentExtension();

const _ = Gettext.domain(Me.metadata['gettext-domain']).gettext;

function getPreferences(group) {
    let prefGroup = null;
    switch (group) {
    case 'generally':
        prefGroup = {
            title: _('Generally'),
            prefs: [
                {
                    type: 'switcher',
                    key: 'show-fahrenheit',
                    label: _('Show temperature in Fahrenheit'),
                    tooltip: _('When this option is activated, Fahrenheit is used as the temperature unit for the display instead of Celsius.'),
                },
                {
                    type: 'combo',
                    key: 'refresh-interval',
                    label: _('Update interval in seconds'),
                    tooltip: _('Sets the update interval for the temperature query of the sensors in seconds.'),
                    values: {
                        '10': '10',
                        '15': '15',
                        '30': '30',
                        '60': '60',
                        '120': '120',
                        '300': '300',
                        '600': '600',
                    },
                },
            ],
        };
        break;
    default: // Nothing to do
    }
    return prefGroup;
}

const TemperaturePreferences = GObject.registerClass(
class TemperaturePreferences extends Gtk.Notebook {
    _init() {
        super._init({
            scrollable: true,
            margin: 15,
            width_request: 720,
            height_request: 480,
        });
        this._settings = ExtensionUtils.getSettings();
        this._createPrefGroup(getPreferences('generally'));
        this.show_all();
    }

    _createPrefGroup(group) {
        let prefGroup = new Gtk.Box({
            orientation: Gtk.Orientation.VERTICAL,
            margin: 15,
            spacing: 15,
        });
        group.prefs.forEach(pref => {
            let prefWidget = this._createPrefWidget(pref);
            if (prefWidget) prefGroup.add(prefWidget);
        });
        this.append_page(prefGroup, new Gtk.Label({ label: group.title }));
    }

    _createPrefWidget(pref) {
        let prefAction = null;
        let prefWidget = null;
        let settings = pref.schema ? new Gio.Settings({ schema_id: pref.schema }) : this._settings;
        switch (pref.type) {
        case 'combo':
            prefAction = new Gtk.ComboBoxText({
                valign: Gtk.Align.CENTER,
                width_request: 100,
            });
            for (let value in pref.values) {
                prefAction.append(value, pref.values[value]);
            }
            prefAction.set_active_id(settings.get_string(pref.key));
            prefAction.connect('changed', () => {
                settings.set_string(pref.key, prefAction.get_active_id());
            });
            settings.connect('changed::%s'.format(pref.key), () => {
                prefAction.set_active_id(settings.get_string(pref.key));
            });
            break;
        case 'switcher':
            prefAction = new Gtk.Switch({
                active: settings.get_boolean(pref.key),
                valign: Gtk.Align.CENTER,
            });
            settings.bind(pref.key, prefAction, 'active', Gio.SettingsBindFlags.DEFAULT);
            break;
        default: // Nothing to do
        }
        if (prefAction) {
            if (pref.tooltip) prefAction.set_tooltip_text(pref.tooltip);
            prefWidget = new Gtk.Box({ orientation: Gtk.Orientation.HORIZONTAL });
            prefWidget.add(new Gtk.Label({
                label: pref.label,
                hexpand: true,
                xalign: 0,
            }));
            prefWidget.add(prefAction);
        }
        return prefWidget;
    }
});

function init() {
    ExtensionUtils.initTranslations();
}

function buildPrefsWidget() {
    return new TemperaturePreferences();
}
